import qrcode, json

def generate_qr_boolean_array(text):
    """
    Generates a QR code for the given text using version 5 and error correction level Q,
    then converts it to a 2D boolean array where each element represents a module (square):
    - True for black (dark) modules
    - False for white (light) modules
    
    Note: QR version 5 has a fixed size of 37x37 modules (excluding borders).
    If the text is too long, it may not fit; adjust version or error correction as needed.
    """
    # Create QR code object with specified parameters
    qr = qrcode.QRCode(
        version=5,  # Fixed to version 5
        error_correction=qrcode.constants.ERROR_CORRECT_Q,  # Q level error correction
        box_size=1,  # Each module is 1 pixel (not relevant for boolean array)
        border=0,  # No border to get raw 37x37 modules
    )
    
    # Add data and generate the QR code
    qr.add_data(text)
    qr.make(fit=True)  # This will attempt to fit, but version is fixed to 5
    
    # Generate the image (black and white)
    img = qr.make_image(fill_color="black", back_color="white")
    
    # Get image dimensions (should be 37x37 for version 5 with border=0)
    width, height = img.size
    
    # Convert to 2D boolean list
    boolean_array = []
    for y in range(height):
        row = []
        for x in range(width):
            pixel = img.getpixel((x, y))
            # Check if pixel is black (0 for grayscale, or (0,0,0) for RGB)
            if pixel == 0 or pixel == (0, 0, 0):
                row.append(True)  # Black module
            else:
                row.append(False)  # White module
        boolean_array.append(row)
    
    return boolean_array

# Example usage:
print(json.dumps(generate_qr_boolean_array("Hello, world!")))